/************************************************************************
 * NAME:	fileio.c
 *
 * DESCR:	Implements the COCO fileio routines.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
#include <stdio.h>
#include "coco.h"

/************************************************************************
 * NAME:	coco_fileio_init()
 *
 * DESCR:	Initialize the fileio structure within the cocofs struct.
 *		The given parameter is the number of files to have open
 *		at any one time.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
coco_fileio_init(struct cocofs *cocofs, int count)
{
    int i;

    cocofs->filebuf = (struct coco_file *)malloc(count * sizeof(struct coco_file));
    cocofs->filecount = count;

    if (cocofs->filebuf == NULL) {
	return(FALSE);
    }

    for (i=0; i < cocofs->filecount; i++) {
	cocofs->filebuf[i].cocofs = cocofs;
	cocofs->filebuf[i].inuse = FALSE;
	cocofs->filebuf[i].bufsize = COCO_SECTORS_PER_GRAN * COCO_SECTOR_SIZE;
	cocofs->filebuf[i].buffer = (char *)malloc(cocofs->filebuf[i].bufsize);
	if (cocofs->filebuf[i].buffer == NULL) {
	    return(FALSE);
	}
    }

    return(TRUE);
}

void
coco_fileio_cleanup(struct cocofs *cocofs)
{
    int	i;

    for (i=0; i < cocofs->filecount; i++) {
	free(cocofs->filebuf[i].buffer);
    }

    free(cocofs->filebuf);
}

/************************************************************************
 * NAME:	coco_file_open()
 *
 * DESCR:	Open the given filename.
 *
 * ARGS:	
 *
 * RETURNS:	a pointer to the file structure
 *
 * NOTES:	- yup, very simple
 ************************************************************************/
struct coco_file *
coco_file_open(struct cocofs *cocofs, char *name)
{
    int i;

    for (i=0; i < cocofs->filecount; i++) {
	struct coco_file	*fptr = &cocofs->filebuf[i];

	if (!fptr->inuse) {
	    fptr->inuse = TRUE;
	    fptr->writemode = FALSE;
	    fptr->cursor = 0;
	    fptr->curgran_no = 0;
	    fptr->curgran = COCO_NULL_GRAN;
	    fptr->cursize = 0;
	    fptr->inode = coco_dir_find(cocofs,name);

	    if (fptr->inode == COCO_INODE_NULL) {
		return((struct coco_file *)NULL);
	    }

	    fptr->firstgran = coco_dirent(cocofs,fptr->inode)->firstgran;
	    fptr->lastsectors = 0;	/* unknown until last FAT	*/
	    fptr->lastbytes = coco_dirent(cocofs,fptr->inode)->lastsecbytes;

	    return(fptr);
	}
    }

    return((struct coco_file *)NULL);
}

/************************************************************************
 * NAME:	coco_file_new()
 *
 * DESCR:	Create a new file that can be written to.  It is assumed
 *		they writing will occur.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
struct coco_file *
coco_file_new(struct cocofs *cocofs, char *name, int filetype, int ascii)
{
    int i;

    for (i=0; i < cocofs->filecount; i++) {
	struct coco_file	*fptr = &cocofs->filebuf[i];

	if (!fptr->inuse) {
	    fptr->inuse = TRUE;
	    fptr->writemode = TRUE;
	    fptr->cursor = 0;
	    fptr->curgran_no = 0;
	    fptr->curgran = COCO_NULL_GRAN;
	    fptr->cursize = 0;
	    fptr->inode = coco_dir_create(cocofs,name,filetype,ascii);

	    if (fptr->inode == COCO_INODE_NULL) {
		return((struct coco_file *)NULL);
	    }

	    fptr->firstgran = 0;
	    fptr->lastsectors = 0;
	    fptr->lastbytes = 0;

	    return(fptr);
	}
    }

    return((struct coco_file *)NULL);
}


void    
coco_file_close(struct coco_file *file)
{
    if (file->inuse) {
	if (file->writemode) {
	    coco_file_flush(file);
	    file->lastbytes = file->cursor % COCO_SECTOR_SIZE;
	    coco_dir_update_entry(file->cocofs,
				  file->inode,
				  file->firstgran,
				  file->curgran,
				  file->lastsectors,
				  file->lastbytes);
	}
	file->inuse = FALSE;
    }
}

/************************************************************************
 * NAME:	coco_file_read()
 *
 * DESCR:	Reads the given number of bytes from the open file.
 *
 * ARGS:	
 *
 * RETURNS:	the number of bytes read, 0 upon EOF.  -1 upon error
 *		if the number of byts returned is less than the number
 *		requested, that PROBABLY represents the last bytes in
 *		the file.  However, someone else could have written
 *		such that more bytes MAY be available upon the next call.
 *		If zero is returned, then it is definitely EOF.
 *
 * NOTES:
 ************************************************************************/
int
coco_file_read(struct coco_file *file, char *buffer, int count)
{
    int	original_count = count;
    int remaining;
    int	toxfer;

    if (!file->inuse) {
	return(-1);
    }

    if (file->writemode) {
	return(-1);
    }

    while (count) {
	remaining = file->cursize - file->cursor;

	if (remaining > 0) {
	    toxfer = MIN(remaining,count);

	    memcpy(buffer,file->buffer+file->cursor,toxfer);	/* copy what we can	*/

	    count -= toxfer;
	    file->cursor += toxfer;
	    buffer += toxfer;

	} else {

	    /* need to load another buffer and try again	*/

	    if (file->curgran != COCO_NULL_GRAN && COCO_FAT_LAST_GRAN(file->cocofs,file->curgran)) {
		break;			/** EOF hit in previous call	*/
	    }

	    file->curgran = (file->curgran == COCO_NULL_GRAN)? file->firstgran : COCO_FAT_NEXT_GRAN(file->cocofs,file->curgran);

	    if (!coco_getgran(file->cocofs, file->curgran, file->buffer)) {	/* get the whole gran	*/
		return(-1);
	    }

	    if (COCO_FAT_LAST_GRAN(file->cocofs,file->curgran)) {
		file->cursize = (COCO_FAT_LAST_SECTORS(file->cocofs,file->curgran)-1) * COCO_SECTOR_SIZE;
		file->cursize += file->lastbytes;
	    } else {
		file->cursize = file->bufsize;
	    }

	    file->cursor = 0;
	    file->curgran_no++;
	}
    }

    return(original_count - count);
}

/************************************************************************
 * NAME:	coco_file_write()
 *
 * DESCR:	Writes the given number of bytes to the open file.
 *
 * ARGS:	
 *
 * RETURNS:	the number of bytes written.  If the number of bytes written
 *		is less than that requested, it means that no more file
 *		space was available. -1 is returned upon error.  0 is
 *		returned if there was no more space to write anything.
 *
 * NOTES:	- 
 ************************************************************************/
int
coco_file_write(struct coco_file *file, char *buffer, int count)
{
    int	original_count = count;
    int remaining;
    int	toxfer;
    int	newgran;

    if (!file->inuse) {
	return(-1);
    }

    if (!file->writemode) {
	return(-1);
    }

    while (count) {

	/* in the case of writing file->cursor points to the next position	*/
	/* to write to.  file->cursize is always bufsize for writing.		*/

	remaining = file->cursize - file->cursor;

	if (remaining > 0) {

	    toxfer = MIN(remaining,count);
	    memcpy(file->buffer+file->cursor,buffer,toxfer);	/* copy what we can	*/
	    count -= toxfer;
	    file->cursor += toxfer;
	    buffer += toxfer;

	} else {

	    /* at this point we have no space left, but more to write	*/

	    coco_file_flush(file);		/* flush out current data	*/

	    /* we need to allocate a new gran for writing	*/

	    newgran =  coco_fat_gran_alloc(file->cocofs);

	    if (newgran == COCO_NULL_GRAN) {			/* ran out of space		*/
		break;
	    }

	    /* we have a new gran, so link it into the chain			*/

	    if (file->curgran == COCO_NULL_GRAN) {
		file->firstgran = newgran;
	    } else {
		coco_fat_gran_link(file->cocofs,file->curgran,newgran);
	    }

	    file->cursize = file->bufsize;
	    file->curgran = newgran;

	    file->cursor = 0;
	    file->curgran_no++;

	}
    }

    return(original_count - count);
}

/************************************************************************
 * NAME:	coco_file_flush()
 *
 * DESCR:	Flush the current file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- decides if a complete gran should be written or part
 *		  of one.  This partial gran writing probably isn't
 *		  COMPLETELY necessary, but what the heck?
 *		- if the file->curgran is zero, this is a no-op.
 ************************************************************************/
int
coco_file_flush(struct coco_file *file)
{
    /* special case zero bytes	*/

    if (file->cursor == 0) {
	file->lastsectors = 1;
    } else {
	file->lastsectors = (file->cursor - 1) / COCO_SECTOR_SIZE + 1;
    }

    if (file->curgran != COCO_NULL_GRAN) {
	return(coco_putgran(file->cocofs,file->curgran,file->buffer,file->lastsectors));
    } else {
	return(TRUE);
    }

}

